
/* Problem G: Gates */
/* Model solution */
/* Author: Krzysztof Sobusiak <sobi@mimuw.edu.pl> */
/* Date: 2001/10/29 */

/*

The original solution algorithm by Grzegorz Jakacki <dzhakakis@yahoo.com>.

We are dealing with unary function $g(x)=f(xx\ldots x)$, where
$f$ is a function implemented by the NAND-gates net.

Assume $f(0^n)=f(1^n)$. Then we can take $g(x)=f(0^n)$.

Assume $f(0^n) \neq f(1^n)$. Take the domain of $f$ partially
ordered by compositewise order, i.e.

$$
x_1 \ldots x_n < y_1 \ldots y_n \iff \forall_i x_i < y_i
$$

Take any maximal chain $c_0, \ldots, c_n$ in this set.
It starts with $0^n$ and
ends with $1^n$ and its consecutive elements differ only in one bit.
Consider the image of this chain through $f$. Assume
there is $k$ such that $0 \le k < n$ and $f(c_k) = f(0^n)$ and
$f(c_{k+1}) = f(1^n)$. $c_k$ and $c_{k+1}$ differ in exactly one bit $l$.
Thus $g(x) = f(c_0c_1\ldots c_{l-1}xc_{l+1}\ldots c_n)$.
On the other hand, if there is no such $k$, then for all k
such that $0 \le k < m$ we must have $f(c_k) = f(c_{k+1})$,
so for all $k$ we have $f(c_k) = f(0^n)$, contradiction.

Observe, that according to the above the minimal number of inputs
to which $x$ has to be connected is zero or one.

The model implementation first checks if $f(0^n) = f(1^n)$. If not, then it
performs a binary search and compute the values of $f(0^k 1^{n-k})$ until
the $k$ is reached, for which $f(0^k 1^{n-k})=f(1^n)$.

The space complexity is $O(n)$. The time complexity is $O(n \log n)$.

*/

#include <stdio.h>

#define MAX_M 200000

FILE *fin, *fout;
int d, n, m;
int val0, val1;
int i, j, x;

struct Cell {
  int a, b;
  int val;
} cells[MAX_M+1];

/* compute the value of the output of the net */
/* for the sequence $0^k 1^{n-k}$ */
int computeValue(int k) {
  int a, b, va, vb, i;
  for (i = 1; i <= m; i++) {
    a = cells[i].a;
    b = cells[i].b;
    va = a < 0 ? -a > k : cells[a].val;
    vb = b < 0 ? -b > k : cells[b].val;
    cells[i].val = !(va && vb);
  }
  return cells[m].val;
}

/* find $k$ such that */
/* $f(0^k 1^{n-k}) = f(0^n)$, $f(0^{k-1} 1^{n-k+1}) = f(1^n)$ */
int search() {
  int p, q, r, v;
  p = 1;
  q = n;
  while (p < q) {
    r = (p + q) / 2;
    v = computeValue(n - r);
    if (v == val1)
      q = r;
    else
      p = r + 1;
  }
  return n - p + 1;
}

int main() {
/*  fin = fopen("cells.in", "r");
  fout = fopen("cells.out", "w");*/
  fin = stdin;
  fout = stdout;
  fscanf(fin, "%d", &d);
  for (i = 0; i < d; i++) {
    fscanf(fin, "%d %d", &n, &m);
    for (j = 1; j <= m; j++)
      fscanf(fin, "%d %d", &cells[j].a, &cells[j].b);
    val0 = computeValue(n);
    val1 = computeValue(0);
    if (val0 == val1)
      for (j = 1; j <= n; j++)
	fprintf(fout, "0");
    else {
      x = search();
      for (j = 1; j < x; j++)
	fprintf(fout, "0");
      fprintf(fout, "x");
      for (j = x + 1; j <= n; j++)
	fprintf(fout, "1");
    }
    fprintf(fout, "\n");
  }
  return 0;
}
